"""Utilities for importing and exporting tests as Word documents."""
from __future__ import annotations

import io
import re
from dataclasses import dataclass
from typing import BinaryIO

from docx import Document

from .models import Answer, GradeCriteria, Question, Test

TITLE_PREFIXES = ("Тест:", "Test:")
DESCRIPTION_PREFIXES = ("Описание:", "Description:")
ACTIVE_PREFIXES = ("Активен:", "IsActive:")
SHUFFLE_QUESTIONS_PREFIXES = ("Перемешивать вопросы:", "ShuffleQuestions:")
SHUFFLE_ANSWERS_PREFIXES = ("Перемешивать ответы:", "ShuffleAnswers:")
TIME_LIMIT_PREFIXES = ("Лимит времени:", "Time limit:", "Time Limit:")
TYPE_PREFIXES = ("Тип:", "Type:")
TEXT_PREFIXES = ("Текст:", "Text:", "Вопрос:")
ANSWERS_MARKERS = {"Ответы:", "Answers:"}
CRITERIA_MARKERS = {"Критерии:", "Criteria:"}
QUESTION_HEADER_PATTERN = re.compile(r"(?:вопрос|question)\s*\d+\s*:", re.IGNORECASE)
CRITERIA_PATTERNS = [
    re.compile(r"-\s*Оценка\s*(\d+)\s*:\s*Максимум ошибок\s*=\s*(\d+)", re.IGNORECASE),
    re.compile(r"-\s*Grade\s*(\d+)\s*:\s*MaxErrors\s*=\s*(\d+)", re.IGNORECASE),
]

QUESTION_TYPE_TO_EXPORT = {
    "single": "одиночный",
    "multiple": "множественный",
}
QUESTION_TYPE_FROM_DOC = {
    "single": "single",
    "multiple": "multiple",
    "одиночный": "single",
    "одиночный выбор": "single",
    "single choice": "single",
    "множественный": "multiple",
    "множественный выбор": "multiple",
    "multiple choice": "multiple",
}

TRUE_VALUES = {"yes", "true", "1", "да", "y"}


class DocumentParseError(Exception):
    """Raised when a Word document cannot be parsed into a test."""


@dataclass
class ParsedAnswer:
    text: str
    is_correct: bool


@dataclass
class ParsedQuestion:
    question_text: str
    question_type: str
    answers: list[ParsedAnswer]


@dataclass
class ParsedTest:
    title: str
    description: str | None
    is_active: bool
    shuffle_questions: bool
    shuffle_answers: bool
    questions: list[ParsedQuestion]
    criteria: list[GradeCriteria]
    time_limit_minutes: int | None


def export_test_to_docx(test: Test) -> io.BytesIO:
    """Serialize a test into a .docx document using Russian labels."""
    document = Document()
    document.add_paragraph(f"Тест: {test.title}")
    document.add_paragraph(f"Описание: {test.description or ''}")
    document.add_paragraph(f"Активен: {'да' if test.is_active else 'нет'}")
    document.add_paragraph(f"Перемешивать вопросы: {'да' if test.shuffle_questions else 'нет'}")
    document.add_paragraph(f"Перемешивать ответы: {'да' if test.shuffle_answers else 'нет'}")
    document.add_paragraph(
        f"Лимит времени: {test.time_limit_minutes if test.time_limit_minutes is not None else ''}"
    )
    document.add_paragraph("")

    for index, question in enumerate(test.questions, start=1):
        document.add_paragraph(f"Вопрос {index}:")
        question_type_label = QUESTION_TYPE_TO_EXPORT.get(question.question_type, question.question_type)
        document.add_paragraph(f"Тип: {question_type_label}")
        document.add_paragraph(f"Текст: {question.question_text}")
        document.add_paragraph("Ответы:")
        for answer in question.answers:
            prefix = "- [x]" if answer.is_correct else "- [ ]"
            document.add_paragraph(f"{prefix} {answer.answer_text}")
        document.add_paragraph("")

    criteria = list(sorted(test.grade_criteria, key=lambda c: c.grade, reverse=True))
    if criteria:
        document.add_paragraph("Критерии:")
        for item in criteria:
            document.add_paragraph(f"- Оценка {item.grade}: Максимум ошибок={item.max_errors}")

    stream = io.BytesIO()
    document.save(stream)
    stream.seek(0)
    return stream


def parse_test_from_docx(source: BinaryIO) -> ParsedTest:
    """Parse a .docx document and return the test definition."""
    document = Document(source)
    title: str | None = None
    description: str | None = None
    is_active = True
    shuffle_questions = False
    shuffle_answers = False
    questions: list[ParsedQuestion] = []
    criteria: list[GradeCriteria] = []
    time_limit_minutes: int | None = None

    current_question: ParsedQuestion | None = None
    in_answers = False
    in_criteria = False

    for paragraph in document.paragraphs:
        text = paragraph.text.strip()
        if not text:
            continue

        if value := _extract_prefixed_value(text, TITLE_PREFIXES):
            title = value
            continue
        if value := _extract_prefixed_value(text, DESCRIPTION_PREFIXES):
            description = value or None
            continue
        if text.startswith(tuple(ACTIVE_PREFIXES)):
            is_active = _parse_bool(text)
            continue
        if text.startswith(tuple(SHUFFLE_QUESTIONS_PREFIXES)):
            shuffle_questions = _parse_bool(text)
            continue
        if text.startswith(tuple(SHUFFLE_ANSWERS_PREFIXES)):
            shuffle_answers = _parse_bool(text)
            continue
        if value := _extract_prefixed_value(text, TIME_LIMIT_PREFIXES):
            value = value.strip()
            if value:
                try:
                    parsed_limit = int(value)
                except ValueError:
                    raise DocumentParseError("Лимит времени должен быть целым числом минут.")
                if parsed_limit <= 0:
                    raise DocumentParseError("Лимит времени должен быть положительным числом.")
                time_limit_minutes = parsed_limit
            else:
                time_limit_minutes = None
            continue

        if QUESTION_HEADER_PATTERN.match(text):
            if current_question:
                _finalize_question(current_question, questions)
            current_question = ParsedQuestion(question_text="", question_type="single", answers=[])
            in_answers = False
            in_criteria = False
            continue

        if value := _extract_prefixed_value(text, TYPE_PREFIXES):
            if not current_question:
                raise DocumentParseError("Строка 'Тип:' встретилась до объявления вопроса.")
            q_type = QUESTION_TYPE_FROM_DOC.get(value.lower())
            if not q_type:
                raise DocumentParseError(
                    "Тип вопроса должен быть 'одиночный' или 'множественный'."
                )
            current_question.question_type = q_type
            continue

        if value := _extract_prefixed_value(text, TEXT_PREFIXES):
            if not current_question:
                raise DocumentParseError("Строка 'Текст:' встречена вне вопроса.")
            current_question.question_text = value
            continue

        if text in ANSWERS_MARKERS:
            if not current_question:
                raise DocumentParseError("Секция ответов встретилась вне вопроса.")
            in_answers = True
            in_criteria = False
            continue

        if text in CRITERIA_MARKERS:
            if current_question:
                _finalize_question(current_question, questions)
                current_question = None
            in_answers = False
            in_criteria = True
            criteria = []
            continue

        if in_answers and text.startswith("-"):
            if not current_question:
                raise DocumentParseError("Ответы заданы вне вопроса.")
            answer = _parse_answer(text)
            current_question.answers.append(answer)
            continue

        if in_criteria and text.startswith("-"):
            grade_criteria = _parse_criteria(text)
            criteria.append(grade_criteria)
            continue

    if current_question:
        _finalize_question(current_question, questions)

    if not title:
        raise DocumentParseError("Не указано название теста (строка 'Тест: ...').")
    if not questions:
        raise DocumentParseError("Документ не содержит ни одного вопроса.")
    if criteria:
        grade_map: dict[int, GradeCriteria] = {}
        for item in criteria:
            if item.grade in grade_map:
                raise DocumentParseError(f"Критерии для оценки {item.grade} указаны более одного раза.")
            grade_map[item.grade] = item
        missing = {5, 4, 3} - set(grade_map)
        if missing:
            raise DocumentParseError(
                "Не хватает критериев для всех оценок (ожидаются значения для 5, 4 и 3)."
            )
        criteria = [grade_map[5], grade_map[4], grade_map[3]]
    else:
        criteria = [
            GradeCriteria(grade=5, max_errors=0),
            GradeCriteria(grade=4, max_errors=1),
            GradeCriteria(grade=3, max_errors=2),
        ]

    return ParsedTest(
        title=title,
        description=description,
        is_active=is_active,
        shuffle_questions=shuffle_questions,
        shuffle_answers=shuffle_answers,
        questions=questions,
        criteria=criteria,
        time_limit_minutes=time_limit_minutes,
    )


def build_test_from_parsed(parsed: ParsedTest, created_by: int) -> Test:
    """Create a Test SQLAlchemy model from parsed data."""
    test = Test(
        title=parsed.title,
        description=parsed.description,
        created_by=created_by,
        is_active=parsed.is_active,
        shuffle_questions=parsed.shuffle_questions,
        shuffle_answers=parsed.shuffle_answers,
        time_limit_minutes=parsed.time_limit_minutes,
    )

    for parsed_question in parsed.questions:
        question = Question(
            question_text=parsed_question.question_text,
            question_type=parsed_question.question_type,
        )
        correct_count = 0
        for parsed_answer in parsed_question.answers:
            if parsed_answer.is_correct:
                correct_count += 1
            question.answers.append(
                Answer(
                    answer_text=parsed_answer.text,
                    is_correct=parsed_answer.is_correct,
                )
            )
        if not correct_count:
            raise DocumentParseError(
                f"Вопрос '{parsed_question.question_text}' не содержит правильных ответов."
            )
        if parsed_question.question_type == "single" and correct_count != 1:
            raise DocumentParseError(
                f"Вопрос '{parsed_question.question_text}' должен иметь ровно один правильный ответ."
            )
        test.questions.append(question)

    for item in parsed.criteria:
        test.grade_criteria.append(GradeCriteria(grade=item.grade, max_errors=item.max_errors))

    return test


def _extract_prefixed_value(text: str, prefixes: tuple[str, ...]) -> str | None:
    for prefix in prefixes:
        if text.startswith(prefix):
            return text[len(prefix) :].strip()
    return None


def _parse_bool(text: str) -> bool:
    value = text.split(":", 1)[1].strip().lower()
    return value in TRUE_VALUES


def _parse_answer(text: str) -> ParsedAnswer:
    """Parse a line like '- [x] Ответ текста'."""
    match = re.match(r"-\s*\[([xX\s])\]\s*(.+)", text)
    if not match:
        raise DocumentParseError("Неверный формат ответа. Используйте '- [x] текст' или '- [ ] текст'.")
    flag, answer_text = match.groups()
    answer_text = answer_text.strip()
    if not answer_text:
        raise DocumentParseError("Текст ответа не может быть пустым.")
    return ParsedAnswer(text=answer_text, is_correct=flag.lower() == "x")


def _parse_criteria(text: str) -> GradeCriteria:
    """Parse a line with оценка/max errors."""
    for pattern in CRITERIA_PATTERNS:
        match = pattern.match(text)
        if match:
            grade = int(match.group(1))
            max_errors = int(match.group(2))
            break
    else:
        raise DocumentParseError(
            "Неверный формат критерия. Используйте '- Оценка N: Максимум ошибок=K'."
        )
    if grade not in {5, 4, 3}:
        raise DocumentParseError("Критерии допустимы только для оценок 5, 4 и 3.")
    if max_errors < 0:
        raise DocumentParseError("Количество ошибок в критерии не может быть отрицательным.")
    return GradeCriteria(grade=grade, max_errors=max_errors)


def _finalize_question(question: ParsedQuestion, accumulator: list[ParsedQuestion]) -> None:
    if not question.question_text:
        raise DocumentParseError("Один из вопросов не содержит текста (строка 'Текст: ...').")
    if len(question.answers) < 2:
        raise DocumentParseError(f"Вопрос '{question.question_text}' должен иметь минимум два ответа.")
    accumulator.append(question)
